<?php

namespace App\Console\Commands;

use App\Models\RepairInspection;
use App\Models\User;
use Carbon\Carbon;
use Filament\Actions\Action;
use Filament\Notifications\Notification;
use Illuminate\Console\Command;

class SendRepairInspectionReminders extends Command
{
    protected $signature = 'repair:send-reminders {--days=6 : Number of days before exit date to send reminder}';
    protected $description = 'Send reminder notifications for repair inspections approaching exit date based on reminder period';

    public function handle()
    {
        $days = (int) $this->option('days');
        $today = Carbon::now();
        
        $this->info("Looking for repair inspections that need reminders today...");
        
        $repairInspections = RepairInspection::whereNotNull('reminder_period')
            ->whereNotNull('date_of_exit')
            ->whereIn('status', ['success', 'warning', 'cancel']) 
            ->with(['customer', 'unit'])
            ->get();
                        
        if ($repairInspections->isEmpty()) {
            $this->info('No repair inspections found with reminder periods set.');
            return;
        }
        
        $this->info("Checking {$repairInspections->count()} repair inspection(s) for reminder dates...");
        
        $notificationCount = 0;
        $users = User::all(); 
        
        foreach ($repairInspections as $repairInspection) {
            $reminderDate = $this->calculateReminderDate($repairInspection);
            
            if ($this->isReminderDate($reminderDate, $days)) {
                if (!$repairInspection->date_of_exit) {
                    continue;
                }
                $daysUntilExit = $repairInspection->date_of_exit->diffInDays($today);
                
                foreach ($users as $user) {
                    // Send the database notification first
                    $notification = $user->notify(new \App\Notifications\RepairInspectionReminder(
                        $repairInspection, 
                        $daysUntilExit,
                        $repairInspection->reminder_period
                    ));

                    $statusColor = $this->getStatusColor($repairInspection->status);
                    $statusText = ucfirst($repairInspection->status);
                    
                    // Send Filament notification with edit action
                    Notification::make()
                        ->title('Reminder: Exit Date')
                        ->body(
                                "Customer: {$repairInspection->customer->name}\n | " .
                                "Serial Number: {$repairInspection->serial_number_unit_gas_detector}\n | " .
                                "Warranty Claim: {$repairInspection->warranty_claim}\n | " .
                                "Status: {$statusText}"
                            )
                        ->color($statusColor)
                        ->icon($this->getStatusIcon($repairInspection->status))
                        ->actions([
                            Action::make('edit')
                                ->button()
                                ->url(route('filament.admin.resources.repair-inspections.edit', [
                                    'record' => $repairInspection->id,
                                    // 'fromNotification' => 1,
                                    // 'ri' => $repairInspection->id,
                                ]))
                                ->label('Edit')
                        ])
                        ->sendToDatabase($user);
                    
                    $this->info("Sent reminder to {$user->name} for Repair Inspection #{$repairInspection->number_certificate} (Exit: " . optional($repairInspection->date_of_exit)->format('Y-m-d') . ", Reminder: {$reminderDate->format('Y-m-d')})");
                    $notificationCount++;
                }
            }
        }

        $this->sendPendingOverdueReminders($users, $today, $notificationCount);

        $this->info("Successfully sent {$notificationCount} reminder notification(s).");
    }
    
    private function calculateReminderDate(RepairInspection $repairInspection): Carbon
    {
        $reminderPeriod = $repairInspection->reminder_period;
        $dateOfExit = $repairInspection->date_of_exit?->copy() ?? Carbon::now();
        
        switch ($reminderPeriod) {
            case '6_months':
                return $dateOfExit->addMonths(6);
            case '1_year':
                return $dateOfExit->addYear();
            case '2_years':
                return $dateOfExit->addYears(2);
            case '6_years':
                return $dateOfExit->addYears(6);
            default:
                return $dateOfExit;
        }
    }
    
    private function isReminderDate(Carbon $reminderDate, int $days): bool
    {
        $today = Carbon::now();
        $startDate = $reminderDate->copy()->subDays($days);
        $endDate = $reminderDate->copy()->addDays($days);
        
        return $today->between($startDate, $endDate);
    }
    
    private function getStatusColor(string $status): string
    {
        return match($status) {
            'success' => 'success',
            'warning' => 'warning',
            'cancel' => 'danger',
            default => 'gray'
        };
    }
    
    private function getStatusIcon(string $status): string
    {
        return match($status) {
            'success' => 'heroicon-o-check-circle',
            'warning' => 'heroicon-o-exclamation-triangle',
            'cancel' => 'heroicon-o-x-circle',
            default => 'heroicon-o-information-circle'
        };
    }
    
    private function sendPendingOverdueReminders($users, Carbon $today, int &$notificationCount): void
    {
        $pending = RepairInspection::where('status', 'pending')
            ->whereNotNull('date_of_entry')
            // Only current month & year
            ->whereMonth('date_of_entry', $today->month)
            ->whereYear('date_of_entry', $today->year)
            ->with(['customer'])
            ->get();

        foreach ($pending as $repairInspection) {
            $daysPending = $repairInspection->date_of_entry?->diffInDays($today) ?? 0;
            if ($daysPending < 7) {
                continue;
            }

            foreach ($users as $user) {
                $user->notify(new \App\Notifications\PendingRepairInspectionReminder(
                    $repairInspection,
                    $daysPending,
                ));

                Notification::make()
                    ->title('Reminder: Pending Over 7 Days')
                    ->body("Customer: {$repairInspection->customer->name} | Serial Number: {$repairInspection->serial_number_unit_gas_detector} | Tanggal Masuk: " . optional($repairInspection->date_of_entry)->format('Y-m-d') . " | Status: {$repairInspection->status}")
                    ->color('warning')
                    ->icon('heroicon-o-clock')
                    ->actions([
                        Action::make('edit')
                            ->button()
                            ->url(route('filament.admin.resources.repair-inspections.edit', [
                                'record' => $repairInspection->id,
                                'fromNotification' => 1,
                                'ri' => $repairInspection->id,
                            ]))
                            ->label('Edit'),
                    ])
                    ->sendToDatabase($user);

                $this->info("Sent pending reminder to {$user->name} for Repair Inspection #{$repairInspection->number_certificate} (Entry: " . optional($repairInspection->date_of_entry)->format('Y-m-d') . ", Pending: {$daysPending} days)");
                $notificationCount++;
            }
        }
    }
}
