<?php

namespace App\Filament\Admin\Resources\RepairInspections\Schemas;

use App\Models\Customer;
use App\Models\RepairInspection;
use App\Models\Unit;
use App\Models\FirmWire;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\ToggleButtons;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Components\Wizard;
use Filament\Schemas\Components\Wizard\Step;
use Filament\Schemas\Schema;
use Filament\Support\Icons\Heroicon;
use Filament\Tables\Columns\ImageColumn;
use Illuminate\Support\Carbon;

class RepairInspectionForm
{
    public static function configure(Schema $schema): Schema
    {
        return $schema
            ->components([
                Wizard::make([
                    Step::make('Unit Barang')
                        ->description('Masukan informasi unit barang')
                        ->schema([
                            Select::make('customer_id')
                                ->label('Customer')
                                ->relationship('customer', 'name')
                                ->getOptionLabelFromRecordUsing(function ($record) {
                                    $po = ($record->po_number && $record->date_of_po)
                                        ? " – {$record->po_number} - {$record->date_of_po}"
                                        : '';

                                    return "{$record->name}{$po}";
                                })
                                ->searchable()
                                ->preload()
                                ->required()
                                ->createOptionForm([
                                    TextInput::make('name')
                                        ->required(),
                                    TextInput::make('po_number'),
                                    DatePicker::make('date_of_po'),
                                ]),
                            Select::make('unit_id')
                                ->label('Unit')
                                ->relationship('unit', 'name')
                                ->searchable()
                                ->preload()
                                ->required()
                                ->createOptionForm([
                                    TextInput::make('name')
                                        ->required(),
                                ]),
                            TextInput::make('serial_number_unit_gas_detector')
                                ->label('Serial Number Unit')
                                ->required(),
                            TextInput::make('type_firm_wire')
                                ->label('Type Firm Wire')
                                ->autocomplete()
                                ->datalist(function () {
                                    return FirmWire::distinct()->pluck('name')->sort()->toArray();
                                })
                                ->placeholder('Ketik untuk mencari data firm wire...')
                                ->helperText('Otomatis terisi dari daftar firm wires yang ada, atau ketik data baru')
                                ->maxLength(255),
                        ])->columns(2),
                    
                    Step::make('Riwayat')
                        ->description('Tambahkan Riwayat Unit Barang')
                        ->schema([
                            // Select::make('type')
                            //     ->label('Tipe')
                            //     ->options([
                            //         'replacement' => 'Perbaikan',
                            //         'calibration' => 'Kalibrasi',
                            //         'replacement_and_calibration' => 'Perbaikan & Kalibrasi',
                            //         'new_and_calibration' => 'Unit Baru & Kalibrasi',
                            //     ])
                            //     ->required()
                            //     ->default('replacement_and_calibration')
                            //     ->live(),
                            Select::make('date_of_purchase_select')
                                ->label('Pilih Tahun PO')
                                ->options(
                                    array_combine(
                                        range(Carbon::now()->year, 1990),
                                        range(Carbon::now()->year, 1990)
                                    )
                                )
                                // ->default(now())
                                ->live()
                                ->afterStateUpdated(function ($state, callable $set) {
                                    if ($state) {   
                                        $set('date_of_purchase', 'PO/' . \Carbon\Carbon::parse($state)->format('Y'));

                                        $currentUser = Auth::user();
                                        $certificateNumber = static::generateCertificateNumber($currentUser);
                                        $set('number_certificate', $certificateNumber);
                                    }
                                }),
                            
                            DatePicker::make('date_of_entry')
                                ->label('Tanggal Pemeriksaan/Masuk')
                                ->required()
                                ->default(now())
                                ->live()
                                ->afterStateUpdated(function ($state, callable $set) {
                                    if ($state) {
                                        \Carbon\Carbon::setLocale('id');
                                        $set('mounth_of_entry', \Carbon\Carbon::parse($state)
                                            ->translatedFormat('F')); 
                                    }
                                }),
                            TextInput::make('mounth_of_entry')
                                ->label('Bulan Masuk')
                                ->hidden()
                                ->dehydrated(),
                            DatePicker::make('date_of_exit')
                                ->label('Tanggal Keluar')
                                ->live()
                                ->afterStateUpdated(function ($state, callable $set) {
                                    if ($state) {
                                        \Carbon\Carbon::setLocale('id');
                                        $set('mounth_of_exit',\Carbon\Carbon::parse($state)
                                            ->translatedFormat('F'));
                                    }
                                }),
                            TextInput::make('mounth_of_exit')
                                ->label('Bulan Keluar')
                                ->hidden()
                                ->dehydrated(),
                            TextInput::make('number_certificate')
                                ->label('Nomor Sertifikat'),

                            TextInput::make('date_of_purchase')
                                ->label('Tahun PO')    
                                ->dehydrated(),
                                //  ->visible(fn ($get) => in_array($get('type'), ['replacement', 'new_and_calibration', 'calibration', 'replacement_and_calibration'])),
                            TextInput::make('no_po')
                                ->label('Nomor PO')    
                                ->dehydrated(),
                            ToggleButtons::make('status')
                                ->options([
                                    'success' => 'Success',
                                    'warning' => 'Warning',
                                    'cancel' => 'Cancel',
                                    'off' => 'Off',
                                    'pending' => 'Pending',
                                ])
                                ->colors([
                                    'success' => 'success',
                                    'warning' => 'warning',
                                    'cancel' => 'danger',
                                    'off' => 'gray',
                                    'pending' => 'gray',
                                ])
                                ->icons([
                                    'success' => Heroicon::OutlinedCheckCircle,
                                    'warning' => Heroicon::OutlinedExclamationCircle,
                                    'cancel' => Heroicon::OutlinedXCircle,
                                    'pending' => Heroicon::OutlinedClock,
                                    'off' => Heroicon::OutlinedPower,
                                ])->inline()
                                ->label('Status')
                                // ->visible(fn ($get) => in_array($get('type'), ['replacement', 'calibration', 'replacement_and_calibration']))
                                ->columnSpan(2),
                            TextInput::make('so')
                                ->label('SO#'),
                                // ->visible(fn ($get) => in_array($get('type'), ['replacement', 'new_and_calibration', 'replacement_and_calibration'])),
                            RichEditor::make('comment')
                                ->label('Catatan')
                                ->columnSpan(3)
                                 ->extraAttributes([
                                        'class' => 'min-h-[1000px]', 
                                 ]),
                                // ->visible(fn ($get) => in_array($get('type'), ['replacement', 'calibration', 'new_and_calibration', 'replacement_and_calibration'])),
                            TextInput::make('warranty_claim')
                                    ->label('Warranty Claim')
                                    ->placeholder('Calibration Unit'),
                                    // ->visible(fn ($get) => in_array($get('type'), ['replacement', 'replacement_and_calibration'])),
                            DatePicker::make('warranty_start_date')
                                ->label('Dari Tanggal'),
                                // ->visible(fn ($get) => in_array($get('type'), ['replacement', 'replacement_and_calibration'])),
                            DatePicker::make('warranty_end_date')
                                ->label('Sampai Tanggal'),
                                // ->visible(fn ($get) => in_array($get('type'), ['replacement', 'replacement_and_calibration'])),
                            Select::make('reminder_period')
                                ->label('Masa Kalibrasi')
                                ->options([
                                    '6_months' => '6 Bulan',
                                    '1_year' => '1 Tahun',
                                ])
                                ->placeholder('Select reminder period'),
                                // ->helperText('Notification will be sent 6 days before the exit date based on selected period')
                                // ->visible(fn ($get) => in_array($get('type'), ['replacement', 'replacement_and_calibration'])),

                            // TextInput::make('status_penawaran')
                            //     ->label('Status Penawaran'),

                            TextInput::make('old_serial_number_censor')
                                ->label('Old Serial Number Sensor'),
                                // ->visible(fn ($get) => in_array($get('type'), ['replacement', 'replacement_and_calibration'])),
                            TextInput::make('new_serial_number_censor')
                                ->label('New Serial Number Sensor'),
                                // ->visible(fn ($get) => in_array($get('type'), ['replacement', 'replacement_and_calibration'])),
                            TextInput::make('old_serial_number_pcb')
                                ->label('Old Serial Number PCB'),
                                // ->visible(fn ($get) => in_array($get('type'), ['replacement', 'replacement_and_calibration'])),
                            TextInput::make('new_serial_number_pcb')
                                ->label('New Serial Number PCB'),
                                // ->visible(fn ($get) => in_array($get('type'), ['replacement', 'replacement_and_calibration'])),
                            TextInput::make('old_serial_number_battery')
                                ->label('Old Serial Number Battery'),
                                // ->visible(fn ($get) => in_array($get('type'), ['replacement', 'replacement_and_calibration'])),
                            TextInput::make('new_serial_number_battery')
                                ->label('New Serial Number Battery'),
                                // ->visible(fn ($get) => in_array($get('type'), ['replacement', 'replacement_and_calibration'])),
                        ])->columns(4),
                    
                    // Step::make('Additional Information')
                    //     ->description('Additional comments and status')
                    //     ->schema([
                     
                    //         FileUpload::make('image')
                    //             // ->multiple()
                    //             ->label('Upload Image Condition')
                    //             ->disk('public')
                    //             ->directory('images')
                    //             ->imageEditor()
                    //             ->storeFileNamesIn('attachment'),
                    //         TextInput::make('name')     
                    //          ->default(fn () => Auth::user()->name)
                    //             ->label('Nama Sales'),
                            
                    //     ])->columns(2),
                ])
                ->skippable()
                // ->columns(3) 
                ->columnSpanFull()   
                ]);
                // Section::make('Additional Information')
                //     ->schema([
                              
                //             FileUpload::make('image')
                //                 // ->multiple()
                //                 ->label('Upload Image Condition')
                //                 ->disk('public')
                //                 ->directory('images')
                //                 ->imageEditor()
                //                 ->storeFileNamesIn('attachment'),
                //             TextInput::make('name')     
                //              ->default(fn () => Auth::user()->name)
                //                 ->label('Nama Sales'),
                //     ]);
        }
        public static function generateCertificateNumber($user = null): string {
            $currentYear = date('Y');
                $currentMonth = date('n'); 
                
                $romanNumerals = [
                    1 => 'I', 2 => 'II', 3 => 'III', 4 => 'IV', 
                    5 => 'V', 6 => 'VI', 7 => 'VII', 8 => 'VIII',
                    9 => 'IX', 10 => 'X', 11 => 'XI', 12 => 'XII'
                ];
                
                $romanMonth = $romanNumerals[$currentMonth];
                
                $userInitials = 'MY'; 
                if ($user && isset($user->name)) {
                    $nameParts = explode(' ', trim($user->name));
                    $initials = '';
                    foreach ($nameParts as $part) {
                        if (!empty($part)) {
                            $initials .= strtoupper(substr($part, 0, 1));
                        }
                    }
                    $userInitials = $initials ?: 'MY';
                }
                
                $lastCertificate = RepairInspection::whereYear('created_at', $currentYear)
                    ->whereMonth('created_at', $currentMonth)
                    ->orderBy('id', 'desc')
                    ->first();
                
                $sequenceNumber = $lastCertificate ? 
                    (int) substr($lastCertificate->number_certificate, 0, 4) + 1 : 1;
                
                return sprintf('%04d/CTR/%s/%s/%s', $sequenceNumber, $userInitials, $romanMonth, $currentYear);
        }

        public static function generateCertificateNumberAlternative($user = null): string
        {
            $currentYear = date('Y');
            $currentMonth = date('n');
            
            $romanNumerals = [
                1 => 'I', 2 => 'II', 3 => 'III', 4 => 'IV', 
                5 => 'V', 6 => 'VI', 7 => 'VII', 8 => 'VIII',
                9 => 'IX', 10 => 'X', 11 => 'XI', 12 => 'XII'
            ];
            
            $romanMonth = $romanNumerals[$currentMonth];
            
            $userInitials = 'MY'; 
            if ($user && isset($user->name)) {
                $nameParts = explode(' ', trim($user->name));
                $initials = '';
                foreach ($nameParts as $part) {
                    if (!empty($part)) {
                        $initials .= strtoupper(substr($part, 0, 1));
                    }
                }
                $userInitials = $initials ?: 'MY';
            }
            
            $count = RepairInspection::whereYear('created_at', $currentYear)
                ->whereMonth('created_at', $currentMonth)
                ->count();
            
            $sequenceNumber = $count + 1;
            
            return sprintf('%04d/CTR/%s/%s/%s', $sequenceNumber, $userInitials, $romanMonth, $currentYear);
        }
}
