<?php

namespace App\Filament\Admin\Widgets;

use App\Models\RepairInspection;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;
use Filament\Widgets\TableWidget;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Carbon;

class WHistoryOverview extends TableWidget
{

    protected static ?string $heading = 'History Reminder';


    public function table(Table $table): Table
    {
        return $table
            ->query(function (): Builder {
                $query = RepairInspection::query()
                    ->with(['customer', 'unit'])
                    ->whereNotNull('date_of_exit')
                    ->whereNotNull('reminder_period');

                $driver = DB::connection()->getDriverName();
                if ($driver === 'sqlite') {
                    $targetExpr = "julianday(date_of_exit, CASE reminder_period\n                        WHEN '6_months' THEN '+182 days'\n                        WHEN '1_year' THEN '+365 days'\n                        WHEN '2_years' THEN '+730 days'\n                        WHEN '6_years' THEN '+2190 days'\n                        ELSE '+0 days' END)";
                    $nowExpr = "julianday('now')";
                    $query->orderByRaw("ABS(($targetExpr) - $nowExpr) ASC, (($targetExpr) - $nowExpr) ASC");
                } else {
                    $targetExpr = "DATE_ADD(date_of_exit, INTERVAL CASE reminder_period\n                        WHEN '6_months' THEN 182\n                        WHEN '1_year' THEN 365\n                        WHEN '2_years' THEN 730\n                        WHEN '6_years' THEN 2190\n                        ELSE 0 END DAY)";
                    $query->orderByRaw("ABS(DATEDIFF($targetExpr, CURDATE())) ASC, DATEDIFF($targetExpr, CURDATE()) ASC");
                }

                return $query;
            })
            ->columns([
                TextColumn::make('serial_number_unit_gas_detector')
                    ->label('Serial Number')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('unit.name')
                    ->label('Unit')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('reminder')
                    ->label('Reminder')
                    ->getStateUsing(function (RepairInspection $record) {
                        $exit = $record->date_of_exit ? ($record->date_of_exit instanceof \Carbon\Carbon ? $record->date_of_exit : Carbon::parse($record->date_of_exit)) : null;
                        if (!$exit || empty($record->reminder_period)) {
                            return null;
                        }

                        $today = Carbon::now()->startOfDay();
                        $exitDay = $exit->copy()->startOfDay();

                        $periodDays = match ($record->reminder_period) {
                            '6_months' => 182,
                            '1_year' => 365,
                            '2_years' => 730,
                            '6_years' => 2190,
                            default => 0,
                        };

                        $target = $exitDay->copy()->addDays($periodDays);
                        $signedDays = $today->diffInDays($target, false);
                        return (int) $signedDays;
                    })
                    ->formatStateUsing(function ($state) {
                        if ($state === null) {
                            return '-';
                        }
                        $days = abs((int) $state);
                        if ((int) $state < 0) {
                            return '+ ' . $days . ' hari';
                        }
                        if ((int) $state === 0) {
                            return '0 hari';
                        }
                        return '- ' . $days . ' hari';
                    })
                    ->badge()
                    ->color(function ($state) {
                        if ($state === null) {
                            return 'gray';
                        }

                        $preWindow = 6;

                        if ((int) $state > 0) {
                            return 'danger';
                        }

                        if ((int) $state <= 0 && (int) $state >= -$preWindow) {
                            return 'warning';
                        }

                        return 'gray';
                    })
                    ->sortable(),
            ])
            ->paginated([5, 10, 25, 50]);
    }
}
