<?php

namespace App\Filament\Imports;

use App\Models\RepairInspection;
use App\Models\Customer;
use App\Models\Unit;
use App\Models\FirmWire;
use Carbon\Carbon;
use Filament\Actions\Imports\ImportColumn;
use Filament\Actions\Imports\Importer;
use Filament\Actions\Imports\Models\Import;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Number;

class RepairInspectionImporter extends Importer
{
    protected static ?string $model = RepairInspection::class;

    public static function getColumns(): array
    {
        return [
            ImportColumn::make('date_of_entry')
                ->label('Date of Entry')
                ->requiredMapping()
                ->rules(['nullable', 'date'])
                ->castStateUsing(fn ($state) => static::parseDate($state)),
            ImportColumn::make('mounth_of_entry')
                ->label('Month of Entry')
                ->rules(['nullable', 'max:255']),
            ImportColumn::make('date_of_exit')
                ->label('Date of Exit')
                // ->requiredMapping()
                ->rules(['nullable', 'date'])
                ->castStateUsing(fn ($state) => static::parseDate($state)),
            ImportColumn::make('mounth_of_exit')
                ->label('Month of Exit')
                ->rules(['nullable', 'max:255']),
            ImportColumn::make('unit_name')
                ->label('Unit Name')
                ->requiredMapping()
                ->rules(['required', 'max:255'])
                ->fillRecordUsing(fn ($record, $state) => null),
            ImportColumn::make('serial_number_unit_gas_detector')
                ->label('Serial Number Unit Gas Detector')
                ->requiredMapping()
                ->rules(['nullable', 'max:255']),
            ImportColumn::make('type_firm_wire')
                ->label('Type Firm Wire')
                ->rules(['nullable', 'max:255']),
            ImportColumn::make('old_serial_number_censor')
                ->label('Old Serial Number Censor')
                ->rules(['nullable', 'max:255']),
            ImportColumn::make('new_serial_number_censor')
                ->label('New Serial Number Censor')
                ->rules(['nullable', 'max:255']),
            ImportColumn::make('old_serial_number_pcb')
                ->label('Old Serial Number PCB')
                ->rules(['nullable', 'max:255']),
            ImportColumn::make('new_serial_number_pcb')
                ->label('New Serial Number PCB')
                ->rules(['nullable', 'max:255']),
            ImportColumn::make('old_serial_number_battery')
                ->label('Old Serial Number Battery')
                ->rules(['nullable', 'max:255']),
            ImportColumn::make('new_serial_number_battery')
                ->label('New Serial Number Battery')
                ->rules(['nullable', 'max:255']),
            ImportColumn::make('date_of_purchase')
                ->label('Date of Purchase')
                ->rules(['nullable', 'max:255']),
            ImportColumn::make('no_po')
                ->label('No PO')
                ->rules(['nullable', 'max:255']),
            ImportColumn::make('customer_name')
                ->label('Customer Name')
                ->requiredMapping()
                ->rules(['required', 'max:255'])
                ->fillRecordUsing(fn ($record, $state) => null),
            ImportColumn::make('warranty_start_date')
                ->label('Warranty Start Date')
                ->rules(['nullable', 'date'])
                ->castStateUsing(fn ($state) => static::parseDate($state, allowNull: true)),
            ImportColumn::make('warranty_end_date')
                ->label('Warranty End Date')
                ->rules(['nullable', 'date'])
                ->castStateUsing(fn ($state) => static::parseDate($state, allowNull: true)),
            ImportColumn::make('so')
                ->label('SO')
                ->rules(['nullable', 'max:255']),
            ImportColumn::make('warranty_claim')
                ->label('Warranty Claim')
                ->rules(['nullable', 'max:255']),
            ImportColumn::make('number_certificate')
                ->label('Number Certificate')
                ->rules(['nullable', 'max:255']),
            ImportColumn::make('comment')
                ->label('Comment')
                ->rules(['nullable', 'max:255']),
            ImportColumn::make("image")
                ->label('Image')
                ->rules(['nullable', 'max:255']),
            ImportColumn::make('status')
                ->label('Status')
                ->rules(['nullable', 'max:255'])
                ->castStateUsing(function ($state) {
                    if ($state === null || $state === '') {
                        return null;
                    }
                    $normalized = strtolower(trim((string) $state));
                    return match ($normalized) {
                        'ok', 'okay', 'success', 'sukses', 'berhasil' => 'success',
                        'warning', 'warn' => 'warning',
                        'cancel', 'canceled', 'cancelled', 'batal' => 'cancel',
                        'off', 'none', 'mati' => 'off',
                        default => $normalized,
                    };
                }),
            ImportColumn::make('status_penawaran')
                ->label('Status Penawaran')
                ->rules(['nullable', 'max:255']),
            ImportColumn::make('reminder_period')
                ->label('Reminder Period')
                ->rules(['nullable', 'max:255']),
        ];
    }

    public function resolveRecord(): ?RepairInspection
    {
        try {
            // Validate required fields
            if (empty($this->data['customer_name'])) {
                throw new \Exception('Customer name is required');
            }
            
            if (empty($this->data['unit_name'])) {
                throw new \Exception('Unit name is required');
            }

            // Parse dates
            $dateOfEntry = static::parseDate($this->data['date_of_entry']  ?? null, allowNull: true);
            $dateOfExit = static::parseDate($this->data['date_of_exit']  ?? null, allowNull: true);
            $warrantyStartDate = static::parseDate($this->data['warranty_start_date'] ?? null, allowNull: true);
            $warrantyEndDate = static::parseDate($this->data['warranty_end_date'] ?? null, allowNull: true);

            if (!$dateOfEntry) {
                throw new \Exception('Date of entry is required');
            }

            // Normalize status
            $status = $this->data['status'] ?? null;
            if ($status !== null && $status !== '') {
                $status = strtolower(trim((string) $status));
                $status = match ($status) {
                    'ok', 'okay', 'success', 'sukses', 'berhasil' => 'success',
                    'warning', 'warn' => 'warning',
                    'cancel', 'canceled', 'cancelled', 'batal' => 'cancel',
                    'off', 'none', 'mati' => 'off',
                    default => $status,
                };
            }

            // Find or create customer
            $customer = Customer::firstOrCreate(
                ['name' => trim($this->data['customer_name'])],
                ['name' => trim($this->data['customer_name'])]
            );

            // Find or create unit
            $unit = Unit::firstOrCreate(
                ['name' => trim($this->data['unit_name'])],
                ['name' => trim($this->data['unit_name'])]
            );

            // Save to firm_wires table if data exists
            if (!empty($this->data['type_firm_wire'])) {
                FirmWire::firstOrCreate(
                    ['name' => trim($this->data['type_firm_wire'])],
                    ['name' => trim($this->data['type_firm_wire'])]
                );
            }

            // Create record
            $record = new RepairInspection();
            $record->fill([
                'date_of_entry' => $dateOfEntry,
                'mounth_of_entry' => $this->data['mounth_of_entry'] ?? null,
                'date_of_exit' => $dateOfExit ?? null,
                'mounth_of_exit' => $this->data['mounth_of_exit'] ?? null,
                'unit_id' => $unit->id,
                'serial_number_unit_gas_detector' => $this->data['serial_number_unit_gas_detector'] ?? null,
                'type_firm_wire' => $this->data['type_firm_wire'] ?? null,
                'old_serial_number_censor' => $this->data['old_serial_number_censor'] ?? null,
                'new_serial_number_censor' => $this->data['new_serial_number_censor'] ?? null,
                'old_serial_number_pcb' => $this->data['old_serial_number_pcb'] ?? null,
                'new_serial_number_pcb' => $this->data['new_serial_number_pcb'] ?? null,
                'old_serial_number_battery' => $this->data['old_serial_number_battery'] ?? null,
                'new_serial_number_battery' => $this->data['new_serial_number_battery'] ?? null,
                'date_of_purchase' => $this->data['date_of_purchase'] ?? null,
                'no_po' => $this->data['no_po'] ?? null,
                'customer_id' => $customer->id,
                'warranty_start_date' => $warrantyStartDate,
                'warranty_end_date' => $warrantyEndDate,
                'so' => $this->data['so'] ?? null,
                'warranty_claim' => $this->data['warranty_claim'] ?? null,
                'number_certificate' => $this->data['number_certificate'] ?? null,
                'comment' => $this->data['comment'] ?? null,
                'image' => $this->data['image'] ?? null,
                'status' => $status,
                'status_penawaran' => $this->data['status_penawaran'] ?? null,
                'reminder_period' => $this->data['reminder_period'] ?? null,
            ]);

            return $record;
        } catch (\Exception $e) {
            // Log error for debugging
            Log::error('Import error: ' . $e->getMessage(), [
                'data' => $this->data
            ]);
            throw $e;
        }
    }

    protected static function parseDate($value, bool $allowNull = false): ?string
    {
        if ($value === null || $value === '') {
            return $allowNull ? null : null;
        }

        $value = trim((string) $value);

        // Handle Excel serial date numbers
        if (is_numeric($value) && $value > 0) {
            try {
                $date = Carbon::createFromTimestamp(((int) $value - 25569) * 86400);
                return $date->toDateString();
            } catch (\Throwable $e) {
                // Continue to other formats
            }
        }

        // Try various date formats
        $formats = [
            'd-m-Y', 'd/m/Y', 'Y-m-d', 'm/d/Y', 'm-d-Y',
            'd-m-y', 'd/m/y', 
            'd M Y', 'd F Y',
            'Y/m/d', 'Y.m.d',
        ];

        foreach ($formats as $format) {
            try {
                $date = Carbon::createFromFormat($format, $value);
                if ($date !== false) {
                    return $date->toDateString();
                }
            } catch (\Throwable $e) {
                continue;
            }
        }

        // Last resort: try Carbon's parser
        try {
            $date = Carbon::parse($value);
            return $date->toDateString();
        } catch (\Throwable $e) {
            return $allowNull ? null : $value;
        }
    }

    public static function getCompletedNotificationBody(Import $import): string
    {
        $body = 'Your repair inspection import has completed and ' . Number::format($import->successful_rows) . ' ' . str('row')->plural($import->successful_rows) . ' imported.';

        if ($failedRowsCount = $import->getFailedRowsCount()) {
            $body .= ' ' . Number::format($failedRowsCount) . ' ' . str('row')->plural($failedRowsCount) . ' failed to import.';
        }

        return $body;
    }
}