<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Auth;
use Barryvdh\DomPDF\Facade\Pdf as Dompdf;

class RepairInspection extends Model
{
    protected $fillable = [
        "date_of_entry",
        "mounth_of_entry",
        "date_of_exit",
        "mounth_of_exit",
        "unit_id",
        "serial_number_unit_gas_detector",
        "type_firm_wire",
        "old_serial_number_censor",
        "new_serial_number_censor",
        "old_serial_number_pcb",
        "new_serial_number_pcb",
        "old_serial_number_battery",
        "new_serial_number_battery",
        "date_of_purchase",
        "customer_id",
        "warranty_start_date",
        "warranty_end_date",
        "so",
        "warranty_claim",
        "number_certificate",
        "image",
        "comment",
        "status",
        "status_penawaran",
        "reminder_period",
        "name",
        "type"
    ];

    protected $casts = [
        'date_of_entry' => 'date',
        'mounth_of_entry' => 'string',
        'date_of_exit' => 'date',
        'mounth_of_exit' => 'string',
        'date_of_purchase' => 'string',
        'warranty_start_date' => 'date',
        'warranty_end_date' => 'date',
    ];

    public function customer(): BelongsTo
    {
        return $this->belongsTo(Customer::class);
    }

    public function unit(): BelongsTo
    {
        return $this->belongsTo(Unit::class);
    }

    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($model) {
            if (empty($model->number_certificate)) {
                $currentUser = Auth::user();
                $model->number_certificate = static::generateCertificateNumber($currentUser);
            }
        });
    }

    public static function generateCertificateNumber($user = null): string
    {
        $currentYear = date('Y');
        $currentMonth = date('n');
        
        $romanNumerals = [
            1 => 'I', 2 => 'II', 3 => 'III', 4 => 'IV', 
            5 => 'V', 6 => 'VI', 7 => 'VII', 8 => 'VIII',
            9 => 'IX', 10 => 'X', 11 => 'XI', 12 => 'XII'
        ];
        
        $romanMonth = $romanNumerals[$currentMonth];
        
        $userInitials = 'MY'; 
        if ($user && isset($user->name)) {
            $nameParts = explode(' ', trim($user->name));
            $initials = '';
            foreach ($nameParts as $part) {
                if (!empty($part)) {
                    $initials .= strtoupper(substr($part, 0, 1));
                }
            }
            $userInitials = $initials ?: 'MY';
        }
        
        $lastCertificate = static::whereYear('created_at', $currentYear)
            ->whereMonth('created_at', $currentMonth)
            ->orderBy('id', 'desc')
            ->first();
        
        $sequenceNumber = $lastCertificate ? 
            (int) substr($lastCertificate->number_certificate, 0, 4) + 1 : 1;
        
        return sprintf('%04d/CTR/%s/%s/%s', $sequenceNumber, $userInitials, $romanMonth, $currentYear);
    }

    public function generatePDF()
    {
        try {
            $repairInspection = $this->load(['customer', 'unit']);

            $certificate = (string) ($this->number_certificate ?: $this->id);
            $safeCertificate = preg_replace('/[\\\\\/]+/', '-', $certificate);
            $filename = "repair-inspection-{$safeCertificate}.pdf";

            $pdf = Dompdf::loadView('pdf.repair-inspection', [
                'repairInspection' => $repairInspection,
            ])->setPaper('a4', 'portrait');

            return $pdf->stream($filename);
        } catch (\Exception $e) {
            return response('PDF generation failed: ' . $e->getMessage(), 500);
        }
    }

    private static function cleanUtf8(mixed $value): mixed
    {
        if (is_string($value)) {
            if (!mb_check_encoding($value, 'UTF-8')) {
                $value = mb_convert_encoding($value, 'UTF-8', 'auto');
            }
            $converted = @iconv('UTF-8', 'UTF-8//IGNORE', $value);
            return $converted !== false ? $converted : $value;
        }

        if ($value instanceof \Illuminate\Database\Eloquent\Model || $value instanceof \Illuminate\Database\Eloquent\Collection) {

            return $value;
        }

        if (is_array($value)) {
            foreach ($value as $k => $v) {
                $value[$k] = self::cleanUtf8($v);
            }
            return $value;
        }

        if (is_object($value)) {
        
            foreach (get_object_vars($value) as $prop => $v) {
                try {
                    $value->$prop = self::cleanUtf8($v);
                } catch (\Throwable $e) {

                }
            }
            return $value;
        }

        return $value;
    }

    private function sanitizeString($value)
    {
        if (is_null($value)) {
            return 'N/A';
        }
        
        $value = (string) $value;
        
        $value = str_replace(["\0", "\r", "\n", "\t"], ['', ' ', ' ', ' '], $value);
        
        $value = mb_convert_encoding($value, 'UTF-8', 'UTF-8');
        
        $value = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $value);
        
        if (!mb_check_encoding($value, 'UTF-8')) {
            $value = mb_convert_encoding($value, 'UTF-8', 'auto');
        }
        
        $value = @iconv('UTF-8', 'UTF-8//IGNORE', $value);
        
        return $value ?: 'N/A';
    }
}
