<?php

namespace App\Notifications;

use App\Models\RepairInspection;
use Filament\Notifications\Notification;
use Illuminate\Bus\Queueable;
use Illuminate\Notifications\Notification as LaravelNotification;

class RepairInspectionReminder extends LaravelNotification
{
    use Queueable;

    protected $repairInspection;
    protected $daysUntilExit;
    protected $reminderPeriod;

    /**
     * Create a new notification instance.
     */
    public function __construct(RepairInspection $repairInspection, int $daysUntilExit, string $reminderPeriod = null)
    {
        $this->repairInspection = $repairInspection;
        $this->daysUntilExit = $daysUntilExit;
        $this->reminderPeriod = $reminderPeriod;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['database'];
    }

    /**
     * Get the database representation of the notification.
     */
    public function toDatabase(object $notifiable): array
    {
        $statusColor = $this->getStatusColor($this->repairInspection->status);
        $statusText = ucfirst($this->repairInspection->status);
        $reminderPeriodText = $this->getReminderPeriodText($this->reminderPeriod);

        return [
            'repair_inspection_id' => $this->repairInspection->id,
            'number_certificate' => $this->repairInspection->number_certificate,
            'status' => $statusText,
            'customer_id' => $this->repairInspection->customer->id ?? null,
            'customer_name' => $this->repairInspection->customer->name ?? 'Unknown',
            'date_of_exit' => optional($this->repairInspection->date_of_exit)->format('Y-m-d'),
            'days_until_exit' => $this->daysUntilExit,
            'reminder_period' => $reminderPeriodText,
            'title' => 'Reminder: Exit Date',
            'body' => "{$this->repairInspection->customer->name} • Repair Inspection #{$this->repairInspection->number_certificate} • Status '{$statusText}'",
            'color' => $statusColor,
            'icon' => $this->getStatusIcon($this->repairInspection->status)
        ];
    }
    
    /**
     * Get status color for notification
     */
    private function getStatusColor(string $status): string
    {
        return match($status) {
            'success' => 'success',
            'warning' => 'warning',
            'cancel' => 'danger',
            default => 'gray'
        };
    }
    
    /**
     * Get reminder period text
     */
    private function getReminderPeriodText(?string $reminderPeriod): string
    {
        return match($reminderPeriod) {
            '6_months' => '6 Months',
            '1_year' => '1 Year',
            '2_years' => '2 Years',
            '6_years' => '6 Years',
            default => 'Unknown'
        };
    }
    
    /**
     * Get status icon for notification
     */
    private function getStatusIcon(string $status): string
    {
        return match($status) {
            'success' => 'heroicon-o-check-circle',
            'warning' => 'heroicon-o-exclamation-triangle',
            'cancel' => 'heroicon-o-x-circle',
            default => 'heroicon-o-information-circle'
        };
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'repair_inspection_id' => $this->repairInspection->id,
            'number_certificate' => $this->repairInspection->number_certificate,
            'status' => $this->repairInspection->status,
            'customer_id' => $this->repairInspection->customer->id ?? null,
            'date_of_exit' => optional($this->repairInspection->date_of_exit)->format('Y-m-d'),
            'days_until_exit' => $this->daysUntilExit,
            'reminder_period' => $this->reminderPeriod,
            'customer_name' => $this->repairInspection->customer->name ?? 'Unknown',
            'unit_name' => $this->repairInspection->unit->name ?? 'Unknown',
        ];
    }
}
